class SweetAlertManager {
    constructor() {
        this.COOL_DOWN_PERIOD = 30000;
        this.init();
    }

    init() {
        this.bindLinkConfirmations();
        this.bindButtonConfirmations();
        this.injectSmsData();
    }

    injectSmsData() {
        // Inject SMS data into window object for access
        if (typeof smsBalance === 'undefined') {
            console.warn('SMS data not injected. Make sure to inject smsBalance, smsThreshold, smsSettingsUrl');
        }
    }

    bindLinkConfirmations() {
        document.querySelectorAll('a.swal-confirm').forEach(link => {
            link.addEventListener('click', (e) => this.handleConfirmation(e, link));
        });
    }

    bindButtonConfirmations() {
        document.querySelectorAll('button.swal-confirm').forEach(button => {
            button.addEventListener('click', (e) => this.handleConfirmation(e, button));
        });
    }

    async handleConfirmation(e, element) {
        e.preventDefault();

        const isLink = element.tagName === 'A';
        const url = isLink ? element.href : null;
        const formId = element.getAttribute('data-form-id');
        
        const title = element.getAttribute('data-title') || 'Are you sure?';
        let text = element.getAttribute('data-text') || 'Please confirm this action.';
        const icon = element.getAttribute('data-icon') || 'question';
        const confirmText = element.getAttribute('data-confirm-button-text') || 'Yes, proceed!';

        // Replace variables in text
        text = this.replaceVariables(text, element);

        // Slowdown check
        if (!await this.checkSlowdown(element, url || formId)) return;

        // SMS check
        if (element.classList.contains('sms-check')) {
            if (!await this.checkSmsBalance(element)) return;
        }

        // Main confirmation
        const result = await Swal.fire({
            title: title,
            text: text,
            icon: icon,
            showCancelButton: true,
            confirmButtonColor: '#3085d6',
            cancelButtonColor: '#d33',
            confirmButtonText: confirmText,
            cancelButtonText: 'Cancel'
        });

        if (result.isConfirmed) {
            if (isLink) {
                window.location.href = url;
            } else if (formId) {
                document.getElementById(formId).submit();
            }
        }
    }

    replaceVariables(text, element) {
        return text.replace(/{(\w+)}/g, (match, p1) => {
            return element.getAttribute('data-' + p1) || match;
        });
    }

    async checkSlowdown(element, identifier) {
        const now = Date.now();
        const lastClickKey = `last-action-${btoa(identifier)}`;
        const lastClick = localStorage.getItem(lastClickKey);

        if (lastClick && (now - lastClick < this.COOL_DOWN_PERIOD)) {
            await Swal.fire({
                title: 'Slow down!',
                text: 'Please wait a moment before performing this action again.',
                icon: 'info',
                confirmButtonText: 'OK'
            });
            return false;
        }

        localStorage.setItem(lastClickKey, now);
        return true;
    }

    async checkSmsBalance(element) {
        if (typeof smsBalance !== 'undefined' && smsBalance <= smsThreshold) {
            const result = await Swal.fire({
                title: 'Low SMS Balance!',
                html: `Your SMS balance is low (${smsBalance}/${smsThreshold}).<br>Please top up before proceeding.`,
                icon: 'warning',
                showCancelButton: true,
                confirmButtonText: 'Top Up Now',
                cancelButtonText: 'Continue Anyway'
            });

            if (result.isConfirmed && typeof smsSettingsUrl !== 'undefined') {
                window.location.href = smsSettingsUrl;
                return false;
            }
        }
        return true;
    }
}

// Initialize when DOM is loaded
document.addEventListener('DOMContentLoaded', () => {
    window.sweetAlertManager = new SweetAlertManager();
});